<?php
defined('ABSPATH') || exit;

class IranDargah_Gateway_Alerts {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Hook to collect alerts
        add_action('admin_init', [$this, 'collect_gateway_alerts'], 20);
    }
    
    /**
     * Collect all gateway alerts and warnings
     */
    public function collect_gateway_alerts() {
        // This will be called to collect alerts
        // We'll store them in a transient for caching
    }
    
    /**
     * Get all alerts for all gateways
     */
    public static function get_all_alerts() {
        $alerts = [];
        
        if (!class_exists('IranDargah_Settings_Loader')) {
            require_once dirname(__FILE__) . '/settings-loader.php';
        }
        
        $gateways = IranDargah_Settings_Loader::get_all_gateways_data();
        
        foreach ($gateways as $gateway) {
            $gateway_alerts = self::get_gateway_alerts($gateway['id'], $gateway);
            if (!empty($gateway_alerts)) {
                $alerts[$gateway['id']] = $gateway_alerts;
            }
        }
        
        return $alerts;
    }
    
    /**
     * Get alerts for a specific gateway
     */
    public static function get_gateway_alerts($gateway_id, $gateway_data = null) {
        $alerts = [];
        
        // Get gateway settings directly
        $settings = get_option("woocommerce_{$gateway_id}_settings", []);
        
        if (empty($settings['enabled']) || $settings['enabled'] !== 'yes') {
            return $alerts;
        }
        
        // Get gateway title from gateway_data if available, otherwise use settings
        $gateway_title = '';
        if ($gateway_data && isset($gateway_data['title'])) {
            $gateway_title = $gateway_data['title'];
        } elseif (!empty($settings['title'])) {
            $gateway_title = $settings['title'];
        } else {
            $gateway_title = $gateway_id; // Fallback
        }
        
        // Check for sandbox mode
        if (isset($settings['sandbox']) && $settings['sandbox'] === 'yes') {
            $alerts[] = [
                'type' => 'warning',
                'code' => 'sandbox_mode',
                'title' => sprintf(__('درگاه %s در حالت آزمایشی فعال است', 'irandargah'), $gateway_title),
                'message' => __('پرداخت‌های واقعی انجام نخواهند شد. برای استفاده از درگاه واقعی، حالت Sandbox را غیرفعال کنید.', 'irandargah'),
                'gateway_id' => $gateway_id,
                'gateway_title' => $gateway_title,
                'timestamp' => current_time('mysql'),
                'dismissible' => false,
            ];
        }
        
        // Check for missing required fields
        $required_fields = self::get_required_fields($gateway_id);
        foreach ($required_fields as $field_key => $field_name) {
            $field_value = $settings[$field_key] ?? '';
            
            if (empty($field_value)) {
                $alerts[] = [
                    'type' => 'error',
                    'code' => 'missing_field_' . $field_key,
                    'title' => sprintf(__('فیلد الزامی %s در درگاه %s خالی است', 'irandargah'), $field_name, $gateway_title),
                    'message' => sprintf(__('لطفاً %s را در تنظیمات درگاه وارد کنید.', 'irandargah'), $field_name),
                    'gateway_id' => $gateway_id,
                    'gateway_title' => $gateway_title,
                    'field_key' => $field_key,
                    'timestamp' => current_time('mysql'),
                    'dismissible' => false,
                ];
            }
        }
        
        // Gateway-specific checks
        $specific_alerts = apply_filters('irandargah_gateway_alerts_' . $gateway_id, [], $settings);
        if (!empty($specific_alerts)) {
            $alerts = array_merge($alerts, $specific_alerts);
        }
        
        return $alerts;
    }
    
    /**
     * Get required fields for each gateway
     */
    private static function get_required_fields($gateway_id) {
        $required_fields = [
            'WC_ZPal' => [
                'merchantcode' => __('مرچنت کد', 'irandargah'),
            ],
            'WC_Sep_Payment_Gateway' => [
                'merchantcode' => __('کد پذیرنده (Terminal ID)', 'irandargah'),
            ],
            'BankMellat' => [
                'terminal_id' => __('ترمینال ID', 'irandargah'),
                'username' => __('نام کاربری', 'irandargah'),
                'password' => __('کلمه عبور', 'irandargah'),
            ],
            'bankparsian' => [
                'pin' => __('پین', 'irandargah'),
            ],
            'WC_aqayepardakht' => [
                'pin' => __('پین', 'irandargah'),
            ],
            'WC_payping' => [
                'paypingToken' => __('توکن', 'irandargah'),
            ],
            'irankish' => [
                'terminalID' => __('شماره پایانه', 'irandargah'),
            ],
            'pasargad' => [
                'pasargad_terminal_id' => __('شماره ترمینال', 'irandargah'),
            ],
            'melli_pay' => [
                'merchant_id' => __('شماره پذیرنده', 'irandargah'),
            ],
        ];
        
        return $required_fields[$gateway_id] ?? [];
    }
    
    /**
     * Remove WordPress admin notices for gateways
     */
    public static function remove_gateway_admin_notices() {
        // Add JavaScript to hide gateway-related notices
        add_action('admin_footer', function() {
            ?>
            <script>
                (function() {
                    // Function to hide gateway notices
                    function hideGatewayNotices() {
                        var notices = document.querySelectorAll('.notice, .notice-warning, .notice-error');
                        notices.forEach(function(notice) {
                            var text = (notice.textContent || notice.innerText || '').trim();
                            
                            // Patterns to match gateway-related notices
                            var patterns = [
                                /حالت آزمایشی.*Sandbox.*فعال است/i,
                                /در حالت آزمایشی.*Sandbox/i,
                                /مرچنت کد.*خالی است/i,
                                /پین.*خالی است/i,
                                /API.*Key.*خالی/i,
                                /درگاه.*در حالت.*آزمایشی/i,
                                /Sandbox.*فعال است/i,
                                /پرداخت.*آزمایشی/i,
                            ];
                            
                            for (var i = 0; i < patterns.length; i++) {
                                if (patterns[i].test(text)) {
                                    notice.style.display = 'none';
                                    return;
                                }
                            }
                        });
                    }
                    
                    // Run immediately
                    hideGatewayNotices();
                    
                    // Run after a short delay (for dynamically added notices)
                    setTimeout(hideGatewayNotices, 100);
                    setTimeout(hideGatewayNotices, 500);
                    setTimeout(hideGatewayNotices, 1000);
                    
                    // Use MutationObserver to catch notices added later
                    if (typeof MutationObserver !== 'undefined') {
                        var observer = new MutationObserver(function(mutations) {
                            hideGatewayNotices();
                        });
                        
                        var target = document.querySelector('body');
                        if (target) {
                            observer.observe(target, {
                                childList: true,
                                subtree: true
                            });
                        }
                    }
                })();
            </script>
            <?php
        }, 999);
    }
    
    /**
     * Get alerts summary (count by type)
     */
    public static function get_alerts_summary() {
        $all_alerts = self::get_all_alerts();
        $summary = [
            'total' => 0,
            'error' => 0,
            'warning' => 0,
            'info' => 0,
        ];
        
        foreach ($all_alerts as $gateway_alerts) {
            foreach ($gateway_alerts as $alert) {
                $summary['total']++;
                $type = $alert['type'] ?? 'info';
                if (isset($summary[$type])) {
                    $summary[$type]++;
                }
            }
        }
        
        return $summary;
    }
}

IranDargah_Gateway_Alerts::get_instance();
