import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { AlertCircle, AlertTriangle, Info, X } from "lucide-react";
import { useQuery } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { useEffect, useState } from "react";

// Declare WordPress globals
declare global {
  interface Window {
    irandargahData?: {
      ajaxUrl: string;
      restUrl: string;
      nonce: string;
      restNonce: string;
    };
  }
}

interface Alert {
  type: 'error' | 'warning' | 'info';
  code: string;
  title: string;
  message: string;
  gateway_id: string;
  gateway_title: string;
  timestamp: string;
  dismissible: boolean;
  field_key?: string;
}

interface GatewayAlerts {
  [gatewayId: string]: Alert[];
}

const fetchAlerts = async (): Promise<GatewayAlerts> => {
  if (!window.irandargahData) {
    throw new Error('IranDargah data not loaded');
  }

  try {
    const response = await fetch(`${window.irandargahData.restUrl}get-alerts`, {
      method: 'GET',
      headers: {
        'X-WP-Nonce': window.irandargahData.restNonce,
        'Content-Type': 'application/json',
      },
      credentials: 'same-origin',
    });

    if (!response.ok) {
      throw new Error('Failed to fetch alerts');
    }

    const data = await response.json();
    return data.alerts || {};
  } catch (error) {
    console.error('Error fetching alerts:', error);
    return {};
  }
};

const getAlertIcon = (type: string) => {
  switch (type) {
    case 'error':
      return <AlertCircle className="h-5 w-5 text-destructive" />;
    case 'warning':
      return <AlertTriangle className="h-5 w-5 text-warning" />;
    case 'info':
      return <Info className="h-5 w-5 text-info" />;
    default:
      return <Info className="h-5 w-5" />;
  }
};

const getAlertVariant = (type: string): "default" | "destructive" => {
  return type === 'error' ? 'destructive' : 'default';
};

export const GatewayAlerts = () => {
  const [dismissedAlerts, setDismissedAlerts] = useState<Set<string>>(new Set());

  const { data: alerts = {}, isLoading, refetch } = useQuery({
    queryKey: ['gateway-alerts'],
    queryFn: fetchAlerts,
    refetchInterval: 30000, // Refetch every 30 seconds
  });

  useEffect(() => {
    // Load dismissed alerts from localStorage
    const stored = localStorage.getItem('irandargah_dismissed_alerts');
    if (stored) {
      try {
        setDismissedAlerts(new Set(JSON.parse(stored)));
      } catch (e) {
        console.error('Error loading dismissed alerts:', e);
      }
    }
  }, []);

  const handleDismiss = (alertCode: string, gatewayId: string) => {
    const key = `${gatewayId}_${alertCode}`;
    const newDismissed = new Set(dismissedAlerts);
    newDismissed.add(key);
    setDismissedAlerts(newDismissed);
    localStorage.setItem('irandargah_dismissed_alerts', JSON.stringify(Array.from(newDismissed)));
  };

  // Flatten all alerts from all gateways
  const allAlerts: (Alert & { gatewayId: string })[] = [];
  Object.entries(alerts).forEach(([gatewayId, gatewayAlerts]) => {
    (gatewayAlerts as Alert[]).forEach((alert) => {
      const key = `${gatewayId}_${alert.code}`;
      if (!dismissedAlerts.has(key) || !alert.dismissible) {
        allAlerts.push({ ...alert, gatewayId });
      }
    });
  });

  // Group alerts by type
  const alertsByType = {
    error: allAlerts.filter(a => a.type === 'error'),
    warning: allAlerts.filter(a => a.type === 'warning'),
    info: allAlerts.filter(a => a.type === 'info'),
  };

  if (isLoading) {
    return null;
  }

  if (allAlerts.length === 0) {
    return null;
  }

  return (
    <div className="space-y-4 mb-6">
      {/* Error Alerts */}
      {alertsByType.error.length > 0 && (
        <div className="space-y-2">
          {alertsByType.error.map((alert, index) => (
            <Alert
              key={`${alert.gatewayId}_${alert.code}_${index}`}
              variant={getAlertVariant(alert.type)}
              className="border-destructive/50 bg-destructive/10"
            >
              <div className="flex items-start gap-3">
                {getAlertIcon(alert.type)}
                <div className="flex-1">
                  <AlertTitle className="flex items-center gap-2">
                    {alert.title}
                    <Badge variant="outline" className="text-xs">
                      {alert.gateway_title}
                    </Badge>
                  </AlertTitle>
                  <AlertDescription className="mt-1">
                    {alert.message}
                  </AlertDescription>
                </div>
                {alert.dismissible && (
                  <Button
                    variant="ghost"
                    size="sm"
                    className="h-6 w-6 p-0"
                    onClick={() => handleDismiss(alert.code, alert.gatewayId)}
                  >
                    <X className="h-4 w-4" />
                  </Button>
                )}
              </div>
            </Alert>
          ))}
        </div>
      )}

      {/* Warning Alerts */}
      {alertsByType.warning.length > 0 && (
        <div className="space-y-2">
          {alertsByType.warning.map((alert, index) => (
            <Alert
              key={`${alert.gatewayId}_${alert.code}_${index}`}
              variant="default"
              className="border-warning/50 bg-warning/10"
            >
              <div className="flex items-start gap-3">
                {getAlertIcon(alert.type)}
                <div className="flex-1">
                  <AlertTitle className="flex items-center gap-2">
                    {alert.title}
                    <Badge variant="outline" className="text-xs">
                      {alert.gateway_title}
                    </Badge>
                  </AlertTitle>
                  <AlertDescription className="mt-1">
                    {alert.message}
                  </AlertDescription>
                </div>
                {alert.dismissible && (
                  <Button
                    variant="ghost"
                    size="sm"
                    className="h-6 w-6 p-0"
                    onClick={() => handleDismiss(alert.code, alert.gatewayId)}
                  >
                    <X className="h-4 w-4" />
                  </Button>
                )}
              </div>
            </Alert>
          ))}
        </div>
      )}

      {/* Info Alerts */}
      {alertsByType.info.length > 0 && (
        <div className="space-y-2">
          {alertsByType.info.map((alert, index) => (
            <Alert
              key={`${alert.gatewayId}_${alert.code}_${index}`}
              variant="default"
              className="border-info/50 bg-info/10"
            >
              <div className="flex items-start gap-3">
                {getAlertIcon(alert.type)}
                <div className="flex-1">
                  <AlertTitle className="flex items-center gap-2">
                    {alert.title}
                    <Badge variant="outline" className="text-xs">
                      {alert.gateway_title}
                    </Badge>
                  </AlertTitle>
                  <AlertDescription className="mt-1">
                    {alert.message}
                  </AlertDescription>
                </div>
                {alert.dismissible && (
                  <Button
                    variant="ghost"
                    size="sm"
                    className="h-6 w-6 p-0"
                    onClick={() => handleDismiss(alert.code, alert.gatewayId)}
                  >
                    <X className="h-4 w-4" />
                  </Button>
                )}
              </div>
            </Alert>
          ))}
        </div>
      )}
    </div>
  );
};

