import { ExtraGatewayTabs } from "@/components/ExtraGatewayTabs";
import { CreditCard, Settings, Shield } from "lucide-react";
import { useQuery } from "@tanstack/react-query";

declare global {
  interface Window {
    irandargahData?: {
      ajaxUrl: string;
      restUrl: string;
      nonce: string;
      restNonce: string;
    };
  }
}

const ExtraGateways = () => {
  const { data: gateways = [] } = useQuery({
    queryKey: ['extra-gateways'],
    queryFn: async () => {
      if (!window.irandargahData) {
        throw new Error('IranDargah data not loaded');
      }

      const restUrl = `${window.irandargahData.restUrl}get-extra-gateways`;
      
      try {
        const response = await fetch(restUrl, {
          method: 'GET',
          headers: {
            'X-WP-Nonce': window.irandargahData.restNonce,
            'Content-Type': 'application/json',
          },
          credentials: 'same-origin',
        });

        if (response.ok) {
          const data = await response.json();
          if (data.success && Array.isArray(data.gateways)) {
            return data.gateways;
          }
        }
      } catch (restError) {
        console.warn('REST API failed, trying AJAX:', restError);
      }

      const formData = new FormData();
      formData.append('action', 'irandargah_get_extra_gateways');
      formData.append('nonce', window.irandargahData.nonce);

      const response = await fetch(window.irandargahData.ajaxUrl, {
        method: 'POST',
        body: formData,
        credentials: 'same-origin',
      });

      if (!response.ok) {
        throw new Error('Failed to fetch extra gateways');
      }

      const data = await response.json();
      return data.data?.gateways || [];
    },
  });

  const enabledCount = gateways.filter((g: any) => g.enabled === 'yes').length;
  const totalCount = gateways.length;

  return (
    <div className="min-h-screen bg-background">
      <header className="border-b border-border bg-card/30 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-xl gradient-primary glow-primary">
                <CreditCard className="h-6 w-6 text-primary-foreground" />
              </div>
              <div>
                <h1 className="text-xl font-bold text-foreground">درگاه‌های اضافی</h1>
                <p className="text-sm text-muted-foreground">مدیریت و پیکربندی درگاه‌های Gateland</p>
              </div>
            </div>
          </div>
        </div>
      </header>

      <main className="container mx-auto px-6 py-8">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
          <div className="gradient-card rounded-xl p-5 border border-border">
            <div className="flex items-center gap-4">
              <div className="h-12 w-12 rounded-lg bg-primary/10 flex items-center justify-center">
                <CreditCard className="h-6 w-6 text-primary" />
              </div>
              <div>
                <p className="text-2xl font-bold text-foreground">{totalCount}</p>
                <p className="text-sm text-muted-foreground">درگاه موجود</p>
              </div>
            </div>
          </div>
          <div className="gradient-card rounded-xl p-5 border border-border">
            <div className="flex items-center gap-4">
              <div className="h-12 w-12 rounded-lg bg-success/10 flex items-center justify-center">
                <Shield className="h-6 w-6 text-success" />
              </div>
              <div>
                <p className="text-2xl font-bold text-foreground">{enabledCount}</p>
                <p className="text-sm text-muted-foreground">درگاه فعال</p>
              </div>
            </div>
          </div>
        </div>

        <div className="gradient-card rounded-2xl p-6 border border-border">
          <div className="flex items-center gap-3 mb-6">
            <Settings className="h-5 w-5 text-primary" />
            <h2 className="text-lg font-semibold text-foreground">انتخاب و پیکربندی درگاه</h2>
          </div>
          <ExtraGatewayTabs />
        </div>
      </main>
    </div>
  );
};

export default ExtraGateways;
