import { GatewayTabs } from "@/components/GatewayTabs";
import { GatewayAlerts } from "@/components/GatewayAlerts";
import { CreditCard, Settings, Shield } from "lucide-react";
import { useQuery } from "@tanstack/react-query";

// Declare WordPress globals
declare global {
  interface Window {
    irandargahData?: {
      ajaxUrl: string;
      restUrl: string;
      nonce: string;
      restNonce: string;
    };
  }
}

const fetchGateways = async () => {
  if (!window.irandargahData) {
    throw new Error('IranDargah data not loaded');
  }

  const response = await fetch(`${window.irandargahData.restUrl}get-gateways`, {
    method: 'GET',
    headers: {
      'X-WP-Nonce': window.irandargahData.restNonce,
      'Content-Type': 'application/json',
    },
  });

  if (!response.ok) {
    throw new Error('Failed to fetch gateways');
  }

  const data = await response.json();
  return data.gateways || [];
};

const Index = () => {
  const { data: gateways = [] } = useQuery({
    queryKey: ['gateways'],
    queryFn: fetchGateways,
  });

  const enabledCount = gateways.filter((g: any) => g.enabled === 'yes').length;
  const totalCount = gateways.length;

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b border-border bg-card/30 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-xl gradient-primary glow-primary">
                <CreditCard className="h-6 w-6 text-primary-foreground" />
              </div>
              <div>
                <h1 className="text-xl font-bold text-foreground">تنظیمات درگاه پرداخت</h1>
                <p className="text-sm text-muted-foreground">مدیریت و پیکربندی درگاه‌های بانکی</p>
              </div>
            </div>
            <div className="flex items-center gap-3">
              <div className="flex items-center gap-2 px-4 py-2 rounded-lg bg-success/10 text-success">
                <Shield size={18} />
                <span className="text-sm font-medium">اتصال امن</span>
              </div>
            </div>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="container mx-auto px-6 py-8">
        {/* Stats */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
          <div className="gradient-card rounded-xl p-5 border border-border animate-fade-in">
            <div className="flex items-center gap-4">
              <div className="h-12 w-12 rounded-lg bg-primary/10 flex items-center justify-center">
                <CreditCard className="h-6 w-6 text-primary" />
              </div>
              <div>
                <p className="text-2xl font-bold text-foreground">{totalCount}</p>
                <p className="text-sm text-muted-foreground">درگاه موجود</p>
              </div>
            </div>
          </div>
          <div className="gradient-card rounded-xl p-5 border border-border animate-fade-in" style={{ animationDelay: '100ms' }}>
            <div className="flex items-center gap-4">
              <div className="h-12 w-12 rounded-lg bg-success/10 flex items-center justify-center">
                <Shield className="h-6 w-6 text-success" />
              </div>
              <div>
                <p className="text-2xl font-bold text-foreground">{enabledCount}</p>
                <p className="text-sm text-muted-foreground">درگاه فعال</p>
              </div>
            </div>
          </div>
          <div className="gradient-card rounded-xl p-5 border border-border animate-fade-in" style={{ animationDelay: '200ms' }}>
            <div className="flex items-center gap-4">
              <div className="h-12 w-12 rounded-lg bg-warning/10 flex items-center justify-center">
                <Settings className="h-6 w-6 text-warning" />
              </div>
              <div>
                <p className="text-2xl font-bold text-foreground">{totalCount - enabledCount}</p>
                <p className="text-sm text-muted-foreground">نیاز به پیکربندی</p>
              </div>
            </div>
          </div>
        </div>

        {/* Gateway Alerts */}
        <GatewayAlerts />

        {/* Gateway Tabs */}
        <div className="gradient-card rounded-2xl p-6 border border-border">
          <div className="flex items-center gap-3 mb-6">
            <Settings className="h-5 w-5 text-primary" />
            <h2 className="text-lg font-semibold text-foreground">انتخاب و پیکربندی درگاه</h2>
          </div>
          <GatewayTabs />
        </div>

        {/* Footer Note */}
        <div className="mt-8 text-center">
          <p className="text-sm text-muted-foreground">
            برای فعال‌سازی هر درگاه، ابتدا اطلاعات مورد نیاز را از پنل درگاه مربوطه دریافت کنید.
          </p>
        </div>
      </main>
    </div>
  );
};

export default Index;
