<?php
/**
 * Plugin Name: آرمان درگاه
 * Description:  مجموعه درگاه‌های پرداخت ایران (زرین پال + سامان)
 * Version: 1.0.3
 * Author: آژانس خلاقیت صالح 
 * Text Domain: armandargah
 * Domain Path: /languages
 * WC requires at least: 3.0
 * WC tested up to: 10.3.5
 * Requires at least: 5.8
 * Tested up to: 6.7.1
 * Requires PHP: 7.4
 * License: GPLv3 or later
 * License URI: https://www.gnu.org/licenses/gpl-3.0.en.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

if (!defined('IRANDARGAH_PLUGIN_DIR')) {
    define('IRANDARGAH_PLUGIN_DIR', plugin_dir_path(__FILE__));
}

if (!defined('IRANDARGAH_PLUGIN_URL')) {
    define('IRANDARGAH_PLUGIN_URL', plugin_dir_url(__FILE__));
}

// بارگذاری Gateland برای درگاه‌های اضافی (سپال، شپا، دایرکت‌پی و ...) تا در تسویه‌حساب ووکامرس نمایش داده شوند
// Gateland از Illuminate\Database استفاده می‌کند که به افزونهٔ PDO نیاز دارد؛ در صورت نبود PDO در هاست، Gateland لود نمی‌شود تا Fatal Error رخ ندهد
global $irandargah_setup_warnings;
if (file_exists(IRANDARGAH_PLUGIN_DIR . 'gateland/gateland.php')) {
    if (extension_loaded('pdo') && extension_loaded('pdo_mysql')) {
        require_once IRANDARGAH_PLUGIN_DIR . 'gateland/gateland.php';
        // همگام‌سازی درگاه‌های اضافی با Gateland قبل از ثبت درگاه‌ها در ووکامرس
        add_action('woocommerce_loaded', function () {
            if (!class_exists('IranDargah_Settings_Loader')) {
                require_once IRANDARGAH_PLUGIN_DIR . 'admin/inc/settings-loader.php';
            }
            IranDargah_Settings_Loader::sync_all_extra_gateways_to_gateland();
        }, 5);
    } else {
        $irandargah_setup_warnings = $irandargah_setup_warnings ?? [];
        $irandargah_setup_warnings[] = __('افزونه‌های PDO یا PDO_MySQL در سرور فعال نیستند. برای استفاده از درگاه‌های اضافی (سپال، شپا، دایرکت‌پی و ...) لطفاً از پشتیبانی هاست بخواهید این افزونه‌ها را در PHP فعال کنند.', 'irandargah');
    }
}

// نگهداری پیام‌های خطا برای نمایش در ادمین
global $irandargah_setup_warnings;
if (!isset($irandargah_setup_warnings) || !is_array($irandargah_setup_warnings)) {
    $irandargah_setup_warnings = [];
}

/**
 * برخی هاست‌ها کتابخانه Action Scheduler را پیش از تعریف بازهٔ «every_minute»
 * لود نمی‌کنند و وردپرس هنگام زمان‌بندی event اصلی آن (action_scheduler_run_queue)
 * خطای invalid_schedule را لاگ می‌کند. این فیلتر در صورت نبود، آن بازه را
 * اضافه می‌کند تا فعال‌سازی افزونه به خاطر خطای کرون متوقف نشود.
 */
add_filter('cron_schedules', function ($schedules) {
    if (!isset($schedules['every_minute'])) {
        $schedules['every_minute'] = [
            'interval' => 60,
            'display'  => __('Every minute', 'irandargah'),
        ];
    }
    return $schedules;
});

/**
 * بارگذاری امن فایل‌های درگاه؛ در صورت نبود فایل، خطای ادمین نشان می‌دهیم
 * تا افزونه به‌جای Fatal Error، پیام راهنما بدهد.
 */
function irandargah_safe_require($relative_path, $label = '') {
    global $irandargah_setup_warnings;
    $full_path = IRANDARGAH_PLUGIN_DIR . ltrim($relative_path, '/');
    if (file_exists($full_path)) {
        require_once $full_path;
        return true;
    }
    $label_text = $label ?: $relative_path;
    $irandargah_setup_warnings[] = sprintf(
        /* translators: 1: gateway/file label, 2: path */
        __('پرونده %1$s پیدا نشد: %2$s. لطفاً بستهٔ افزونه را کامل و مجدداً بارگذاری کنید.', 'irandargah'),
        $label_text,
        $relative_path
    );
    return false;
}

/**
 * نمایش هشدارهای راه‌اندازی (فایل ناقص یا نبود جدول Action Scheduler)
 */
add_action('admin_notices', function () {
    global $irandargah_setup_warnings;
    if (!current_user_can('manage_options') || empty($irandargah_setup_warnings)) {
        return;
    }
    echo '<div class="notice notice-error"><p>';
    echo implode('<br>', array_map('esc_html', $irandargah_setup_warnings));
    echo '</p></div>';
});

/**
 * چک می‌کند که جداول Action Scheduler ووکامرس ساخته شده باشد؛
 * در صورت نبود، هشدار می‌دهد تا از ابزارهای ووکامرس برای ساخت جداول استفاده شود.
 */
add_action('plugins_loaded', function () {
    global $wpdb, $irandargah_setup_warnings;
    $table = $wpdb->prefix . 'actionscheduler_actions';
    $exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table));
    if ($exists !== $table) {
        $irandargah_setup_warnings[] = __('جداول Action Scheduler در دیتابیس موجود نیست. از ووکامرس > وضعیت > ابزارها، گزینه ساخت جداول برنامه‌ریز را اجرا کنید یا دسترسی CREATE/ALTER دیتابیس را بررسی کنید.', 'irandargah');
    }
});

/**
 * اگر جداول Action Scheduler وجود نداشته باشد، از اجرای آن جلوگیری می‌کنیم
 * تا Fatal Error رخ ندهد و افزونه بتواند فعال شود. پس از ساخت جداول، این
 * فیلتر مانع اجرا نخواهد شد.
 */
add_filter('action_scheduler_pre_init', function ($should_init) {
    global $wpdb, $irandargah_setup_warnings;
    $table = $wpdb->prefix . 'actionscheduler_actions';
    $exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table));
    if ($exists !== $table) {
        $irandargah_setup_warnings[] = __('اجرای Action Scheduler به دلیل نبود جداول متوقف شد. لطفاً جداول را از ووکامرس > وضعیت > ابزارها ایجاد کنید.', 'irandargah');
        return false;
    }
    return $should_init;
}, 5);

if (!function_exists('irandargah_get_gateway_ids')) {
    /**
     * لیست همه کلاس‌های درگاه‌هایی که توسط داشبورد ایران‌درگاه مدیریت می‌شوند
     *
     * این شناسه‌ها همان شناسه‌های مورد استفاده در ووکامرس هستند.
     *
     * @return array
     */
    function irandargah_get_gateway_ids() {
        return [
            'WC_ZPal',                 // زرین‌پال
            'WC_Sep_Payment_Gateway',  // سامان (سپ)
            'WC_aqayepardakht',        // آقای پرداخت
            'WC_payping',              // پی‌پینگ
            'irankish',                // ایران‌کیش
            'bankparsian',             // پارسیان
            'pasargad',                // پاسارگاد
            'melli_pay',               // بانک ملّی
            'BankMellat',              // بانک ملت (شناسه واقعی)
            'WC_Zibal',                // زیبال
        ];
    }
}

/**
 * هنگام فعال‌سازی افزونه: همهٔ درگاه‌ها را پیش‌فرض غیرفعال می‌کنیم
 * و تنظیم enabled آن‌ها در ووکامرس را نیز روی "no" می‌گذاریم تا فقط از طریق داشبورد ایران‌درگاه فعال شوند.
 */
register_activation_hook(__FILE__, function () {
    $gateway_ids = irandargah_get_gateway_ids();
    $statuses = [];
    foreach ($gateway_ids as $gateway_id) {
        $statuses[$gateway_id] = 0;
        $option_name = "woocommerce_{$gateway_id}_settings";
        $gateway_settings = get_option($option_name, []);
        $gateway_settings['enabled'] = 'no';
        update_option($option_name, $gateway_settings);
    }
    update_option('irandargah_gateway_statuses', $statuses);
});

// Declare compatibility with WooCommerce features
add_action('before_woocommerce_init', function () {
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('product_block_editor', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('analytics', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('new_navigation', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('store_api', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('subscriptions', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('subscriptions_update', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('woocommerce-blocks', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('payment_gateway_suggestions', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('order_attribution', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('minified_css', __FILE__, true);
    }
});

// Load Zarinpal gateway (full implementation with Blocks support)
// قبلاً فایل zarinpal/zarinpal.php عملاً غیرفعال شده بود؛
// اینجا مستقیماً bootstrap اصلی زرین‌پال را لود می‌کنیم.
irandargah_safe_require('zarinpal/index.php', 'زرین‌پال');

// Load SEP gateway
irandargah_safe_require('sep/sep.php', 'سپ سامان');

// Load additional Iranian gateways so they can be managed from IranDargah dashboard.
// آن‌ها را روی هوک plugins_loaded لود می‌کنیم تا ابتدا ووکامرس کلاس‌های خودش را بارگذاری کرده باشد.
add_action('plugins_loaded', function () {
    $gateway_bootstraps = [
        'aqayepardakht/index.php',
        'payping/index.php',
        'irankish/class-wc-gateway-irankish.php',
        'parsian/parsian-woocommerce.php',
        'pasargad/woocommerce-pasargad.php',
        'melli/index.php',
        'mellat/index.php',
        'zibal/index.php',
    ];

    foreach ($gateway_bootstraps as $relative_file) {
        $full_path = IRANDARGAH_PLUGIN_DIR . $relative_file;
        if (file_exists($full_path)) {
            require_once $full_path;
        }
    }

    /**
     * برخی درگاه‌ها داخل فایل خودشان، روی هوک plugins_loaded رجیستر می‌شوند.
     * چون ما آن‌ها را همین‌جا (پس از اجرای اولیۀ plugins_loaded) لود می‌کنیم،
     * باید توابع init آن‌ها را دستی فراخوانی کنیم تا فیلترها/کلاس‌ها ثبت شوند.
     */
    $manual_inits = [
        'load_aqayepardakht_woo_gateway',
        'load_payping_woo_gateway',
        'init_melli_woocommerce',
        'Load_BankMellat_Gateway',
        // اطمینان از ثبت درگاه زیبال حتی اگر هوک plugins_loaded آن
        // بعد از اجرای اولیه اضافه شده باشد.
        'load_zibal_woo_gateway',
    ];

    foreach ($manual_inits as $init_fn) {
        if (function_exists($init_fn)) {
            $init_fn();
        }
    }
    
    // اطمینان از ثبت درگاه‌هایی که نیاز به تابع خاصی دارند
    // ایران‌کیش
    if (function_exists('Load_IranKish_Gateway') && class_exists('WC_Payment_Gateway') && !class_exists('WC_Gateway_Irankish')) {
        Load_IranKish_Gateway();
    }
    
    // پاسارگاد
    if (function_exists('woocommerce_pasargad_payment_init') && class_exists('WC_Payment_Gateway')) {
        woocommerce_pasargad_payment_init();
    }
    
    // پارسیان
    if (function_exists('Load_BankParsian_Gateway') && class_exists('WC_Payment_Gateway') && !class_exists('WC_Gateway_Bankparsian')) {
        Load_BankParsian_Gateway();
    }
}, 20);

/**
 * ثبت مرکزی تمام درگاه‌های ایرانی در لیست ووکامرس
 * این فیلتر اطمینان می‌دهد که همه درگاه‌ها در تب "پرداخت" نمایش داده شوند
 */
add_filter('woocommerce_payment_gateways', function($methods) {
    // فقط در پیشخوان این فیلتر را اعمال کن
    if (!is_admin()) {
        return $methods;
    }
    
    // Mapping شناسه درگاه به نام کلاس (برای درگاه‌هایی که ID و نام کلاس متفاوت است)
    $gateway_class_mapping = [
        'irankish' => 'WC_Gateway_Irankish',
        'melli_pay' => 'WC_Gateway_Melli',
        'bankparsian' => 'WC_Gateway_Bankparsian',
        'pasargad' => 'WC_Pasargad_Gateway',
        'BankMellat' => 'WC_BankMellat', // بانک ملت
        'WC_Zibal' => 'WC_Zibal', // زیبال
    ];
    
    $gateway_ids = irandargah_get_gateway_ids();
    
    foreach ($gateway_ids as $gateway_id) {
        // بررسی اینکه آیا درگاه در لیست است
        $found = false;
        $class_name_to_check = isset($gateway_class_mapping[$gateway_id]) ? $gateway_class_mapping[$gateway_id] : $gateway_id;
        
        foreach ($methods as $method) {
            if (is_object($method)) {
                $method_id = method_exists($method, 'get_id') ? $method->get_id() : (isset($method->id) ? $method->id : '');
                $class_name = get_class($method);
                
                // بررسی بر اساس ID درگاه
                if ($method_id === $gateway_id) {
                    $found = true;
                    break;
                }
                
                // بررسی بر اساس نام کلاس
                if ($class_name === $gateway_id || $class_name === $class_name_to_check) {
                    $found = true;
                    break;
                }
            } elseif (is_string($method)) {
                // اگر رشته است، بررسی بر اساس نام کلاس
                if ($method === $gateway_id || $method === $class_name_to_check) {
                    $found = true;
                    break;
                }
            }
        }
        
        // اگر پیدا نشد، سعی کن کلاس را پیدا کنی و اضافه کنی
        if (!$found) {
            // اول سعی کن با نام کلاس اصلی
            if (class_exists($gateway_id)) {
                $methods[] = $gateway_id;
                $found = true;
            }
            // اگر پیدا نشد، با نام کلاس mapped
            elseif (isset($gateway_class_mapping[$gateway_id]) && class_exists($gateway_class_mapping[$gateway_id])) {
                $methods[] = $gateway_class_mapping[$gateway_id];
                $found = true;
            }
        }
    }
    
    return $methods;
}, 999);

// Load custom Iranian Gateways admin dashboard
if (is_admin()) {
    require_once IRANDARGAH_PLUGIN_DIR . 'admin/dashboard.php';
}

// مخفی کردن منوهای نابیک و گیت‌لند در پیشخوان وردپرس (زیرمجموعه‌ها هم مخفی می‌شوند)
add_action('admin_menu', function () {
    remove_menu_page('nabik');
    remove_menu_page('gateland');
}, 999);

/**
 * مسیریابی تنظیمات درگاه‌ها به داشبورد ایران‌درگاه (جلوگیری از تغییر تنظیمات از صفحه ووکامرس)
 */
add_action('admin_init', function () {
    if (!is_admin()) {
        return;
    }
    $gateway_ids = irandargah_get_gateway_ids();
    $section = isset($_GET['section']) ? sanitize_text_field(wp_unslash($_GET['section'])) : '';
    $tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : '';
    $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';

    if ($page === 'wc-settings' && $tab === 'checkout' && $section && in_array($section, $gateway_ids, true)) {
        wp_safe_redirect(admin_url('admin.php?page=irandargah-dashboard'));
        exit;
    }
});

/**
 * همگام‌سازی وضعیت درگاه‌ها از تنظیمات ووکامرس به irandargah_gateway_statuses
 * این تابع در صورت عدم وجود رکورد در irandargah_gateway_statuses، از تنظیمات ووکامرس استفاده می‌کند
 * 
 * این تابع فقط در پیشخوان اجرا می‌شود تا از بار اضافی در فرانت‌اند جلوگیری شود
 */
add_action('admin_init', function() {
    // فقط یک بار در هر درخواست اجرا شود
    static $synced = false;
    if ($synced) {
        return;
    }
    $synced = true;
    
    $statuses = get_option('irandargah_gateway_statuses', []);
    $gateway_ids = irandargah_get_gateway_ids();

    // اگر وضعیت درگاه در irandargah وجود نداشت، پیش‌فرض غیرفعال قرار بده (0)
    foreach ($gateway_ids as $gateway_id) {
        if (!isset($statuses[$gateway_id])) {
            $statuses[$gateway_id] = 0;
        }
    }
    update_option('irandargah_gateway_statuses', $statuses);

    // وضعیت ایران‌درگاه مرجع است؛ آن را به تنظیمات ووکامرس هم اعمال می‌کنیم
    foreach ($gateway_ids as $gateway_id) {
        $option_name = "woocommerce_{$gateway_id}_settings";
        $gateway_settings = get_option($option_name, []);
        $gateway_settings['enabled'] = (int)$statuses[$gateway_id] === 1 ? 'yes' : 'no';
        update_option($option_name, $gateway_settings);
    }
}, 10);

// AJAX handler for toggling gateway status from Iranian Gateways dashboard
add_action('wp_ajax_irandargah_toggle_gateway', function() {
    check_ajax_referer('irandargah_dashboard_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('دسترسی لازم را ندارید.', 'irandargah')]);
    }

    $gateway_id = isset($_POST['gateway_id']) ? sanitize_text_field(wp_unslash($_POST['gateway_id'])) : '';
    $enabled_param = isset($_POST['enabled']) ? sanitize_text_field(wp_unslash($_POST['enabled'])) : '';
    
    // Handle different formats: 'true'/'false', 'yes'/'no', '1'/'0'
    $enabled = false;
    if ($enabled_param === 'yes' || $enabled_param === 'true' || $enabled_param === '1' || $enabled_param === 1) {
        $enabled = true;
    }

    if (empty($gateway_id)) {
        wp_send_json_error(['message' => __('شناسه درگاه معتبر نیست.', 'irandargah')]);
    }

    if (!class_exists('IranDargah_Settings_Loader')) {
        require_once IRANDARGAH_PLUGIN_DIR . 'admin/inc/settings-loader.php';
    }

    IranDargah_Settings_Loader::toggle_gateway($gateway_id, $enabled);

    wp_send_json_success(['message' => __('وضعیت درگاه تغییر کرد.', 'irandargah')]);
});

// AJAX handler for getting all gateways
add_action('wp_ajax_irandargah_get_gateways', function() {
    check_ajax_referer('irandargah_dashboard_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('دسترسی لازم را ندارید.', 'irandargah')]);
    }

    if (!class_exists('IranDargah_Settings_Loader')) {
        require_once IRANDARGAH_PLUGIN_DIR . 'admin/inc/settings-loader.php';
    }

    $gateways = IranDargah_Settings_Loader::get_all_gateways_data();

    wp_send_json_success([
        'gateways' => $gateways ?: [],
    ]);
});

// AJAX handler for saving gateway settings
add_action('wp_ajax_irandargah_save_settings', function() {
    check_ajax_referer('irandargah_dashboard_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('دسترسی لازم را ندارید.', 'irandargah')]);
    }

    $gateway_id = isset($_POST['gateway_id']) ? sanitize_text_field(wp_unslash($_POST['gateway_id'])) : '';
    $settings = isset($_POST['settings']) && is_array($_POST['settings']) ? $_POST['settings'] : [];

    if (empty($gateway_id)) {
        wp_send_json_error(['message' => __('شناسه درگاه معتبر نیست.', 'irandargah')]);
    }

    if (!class_exists('IranDargah_Settings_Loader')) {
        require_once IRANDARGAH_PLUGIN_DIR . 'admin/inc/settings-loader.php';
    }

    IranDargah_Settings_Loader::save_gateway_settings($gateway_id, $settings);

    wp_send_json_success(['message' => __('تنظیمات با موفقیت ذخیره شد.', 'irandargah')]);
});

/**
 * کنترل نمایش درگاه‌ها در فرانت‌اند: تنها درگاه‌هایی که در داشبورد ایران‌درگاه فعال شده‌اند نمایش داده می‌شوند.
 */
add_filter('woocommerce_available_payment_gateways', function($available) {
    if (is_admin()) {
        return $available;
    }
    
    // اگر available خالی است یا آرایه نیست، آن را به آرایه خالی تبدیل کن
    if (!is_array($available)) {
        $available = [];
    }

    // درگاه‌های اضافی (سپال، شپا، دایرکت‌پی و ...): فقط درگاه‌های Gateland که در داشبورد فعال‌اند نمایش داده شوند
    $extra_map = get_option('irandargah_extra_gateway_id_map', []);
    $gateland_id_to_extra = array_flip($extra_map);
    foreach ($available as $key => $gateway) {
        if (strpos($key, 'gateland_') === 0 && $key !== 'gateland') {
            $gateland_id = (int) str_replace('gateland_', '', $key);
            if ($gateland_id > 0 && isset($gateland_id_to_extra[$gateland_id])) {
                $extra_id = $gateland_id_to_extra[$gateland_id];
                $gateway_key = str_replace('extra_', '', $extra_id);
                $settings = get_option('irandargah_extra_' . $gateway_key . '_settings', []);
                if (empty($settings['enabled']) || $settings['enabled'] !== 'yes') {
                    unset($available[$key]);
                }
            }
        }
    }

    $statuses = get_option('irandargah_gateway_statuses', []);
    $gateway_ids = irandargah_get_gateway_ids();
    
    // اطمینان از همگام بودن وضعیت درگاه‌ها (فقط یک بار در هر درخواست)
    static $synced_frontend = false;
    if (!$synced_frontend) {
        $synced_frontend = true;
        foreach ($gateway_ids as $gateway_id) {
            if (!isset($statuses[$gateway_id])) {
                $statuses[$gateway_id] = 0;
            }
            // همگام‌سازی وضعیت از داشبورد به تنظیمات ووکامرس
            $option_name = "woocommerce_{$gateway_id}_settings";
            $gateway_settings = get_option($option_name, []);
            $dashboard_enabled = (int)$statuses[$gateway_id] === 1;
            $wc_enabled = isset($gateway_settings['enabled']) && $gateway_settings['enabled'] === 'yes';
            
            // اگر وضعیت در داشبورد با ووکامرس متفاوت است، آن را همگام کن
            if ($dashboard_enabled !== $wc_enabled) {
                $gateway_settings['enabled'] = $dashboard_enabled ? 'yes' : 'no';
                update_option($option_name, $gateway_settings);
            }
        }
        // بارگذاری مجدد وضعیت‌ها پس از همگام‌سازی
        $statuses = get_option('irandargah_gateway_statuses', []);
    }
    
    // Mapping شناسه درگاه به نام کلاس (برای درگاه‌هایی که ID و نام کلاس متفاوت است)
    $gateway_class_mapping = [
        'irankish' => 'WC_Gateway_Irankish',
        'melli_pay' => 'WC_Gateway_Melli',
        'bankparsian' => 'WC_Gateway_Bankparsian',
        'pasargad' => 'WC_Pasargad_Gateway',
        'BankMellat' => 'WC_BankMellat', // بانک ملت
        'WC_Zibal' => 'WC_Zibal', // زیبال
    ];

    // ابتدا همه درگاه‌های ایران‌درگاه را پیدا کن
    $found_gateways = [];
    foreach ($available as $key => $gateway) {
        if (!is_object($gateway)) {
            continue;
        }
        
        // دریافت ID درگاه از روش‌های مختلف
        $obj_id = '';
        if (method_exists($gateway, 'get_id')) {
            $obj_id = $gateway->get_id();
        } elseif (isset($gateway->id)) {
            $obj_id = $gateway->id;
        }
        
        $class_name = get_class($gateway);
        
        // بررسی اینکه آیا این درگاه در لیست ایران‌درگاه است
        $matched_gateway_id = null;
        foreach ($gateway_ids as $gateway_id) {
            $class_name_to_check = isset($gateway_class_mapping[$gateway_id]) ? $gateway_class_mapping[$gateway_id] : $gateway_id;
            
            // تطابق بر اساس کلید آرایه (اولویت اول)
            if ($key === $gateway_id) {
                $matched_gateway_id = $gateway_id;
                break;
            }
            
            // تطابق بر اساس ID درگاه (اولویت دوم)
            if (!empty($obj_id) && $obj_id === $gateway_id) {
                $matched_gateway_id = $gateway_id;
                break;
            }
            
            // تطابق بر اساس نام کلاس (اولویت سوم)
            if ($class_name === $gateway_id || $class_name === $class_name_to_check) {
                $matched_gateway_id = $gateway_id;
                break;
            }
            
            // تطابق بر اساس نام کلاس که با شناسه درگاه شروع می‌شود (برای موارد خاص)
            if (strpos($class_name, $gateway_id) !== false || strpos($class_name, $class_name_to_check) !== false) {
                $matched_gateway_id = $gateway_id;
                break;
            }
        }
        
        // اگر درگاه در لیست ایران‌درگاه است، آن را ذخیره کن
        if ($matched_gateway_id !== null) {
            $found_gateways[$key] = $matched_gateway_id;
        }
    }

    // حالا درگاه‌های پیدا شده را فیلتر کن
    // فقط درگاه‌هایی که در داشبورد ایران‌درگاه فعال شده‌اند نمایش داده می‌شوند
    // بدون در نظر گرفتن حالت تست یا وضعیت اطلاعات حساب
    foreach ($found_gateways as $key => $gateway_id) {
        if (!isset($available[$key])) {
            continue;
        }
        
        // بررسی وضعیت فعال بودن درگاه در داشبورد ایران‌درگاه
        $is_enabled = isset($statuses[$gateway_id]) && (int)$statuses[$gateway_id] === 1;
        
        // اگر درگاه غیرفعال است، آن را حذف کن
        // در غیر این صورت، درگاه را نگه دار (بدون بررسی حالت تست یا اطلاعات حساب)
        if (!$is_enabled) {
            unset($available[$key]);
        }
    }

    /**
     * اطمینان از نمایش درگاه در حالت آزمایشی:
     * اگر درگاه در داشبورد فعال باشد و sandbox آن در تنظیمات ووکامرس روی yes باشد
     * ولی به هر دلیل توسط افزونه اصلی از لیست روش‌های پرداخت حذف شده باشد،
     * در اینجا به صورت اجباری آن را به لیست available اضافه می‌کنیم.
     */
    foreach ($gateway_ids as $gateway_id) {
        // اگر از قبل در لیست موجود است، نیازی به افزودن دوباره نیست
        $is_already_available = false;
        foreach ($available as $key => $gateway) {
            if (!is_object($gateway)) {
                continue;
            }

            $obj_id = '';
            if (method_exists($gateway, 'get_id')) {
                $obj_id = $gateway->get_id();
            } elseif (isset($gateway->id)) {
                $obj_id = $gateway->id;
            }

            if ($key === $gateway_id || $obj_id === $gateway_id) {
                $is_already_available = true;
                break;
            }
        }

        if ($is_already_available) {
            continue;
        }

        // فقط درگاه‌هایی که در داشبورد فعال شده‌اند
        $is_enabled_in_dashboard = isset($statuses[$gateway_id]) && (int) $statuses[$gateway_id] === 1;
        if (!$is_enabled_in_dashboard) {
            continue;
        }

        // اگر درگاه در داشبورد فعال است، آن را به لیست اضافه کن
        // (بدون نیاز به بررسی sandbox یا اطلاعات حساب)

        // تشخیص نام کلاس درگاه
        $class_name_to_check = isset($gateway_class_mapping[$gateway_id]) ? $gateway_class_mapping[$gateway_id] : $gateway_id;
        
        // استفاده از سیستم WooCommerce برای لود کردن درگاه‌ها
        $gateway_instance = null;
        
        if (function_exists('WC') && WC()->payment_gateways()) {
            $wc_gateways = WC()->payment_gateways()->payment_gateways();
            
            // بررسی اینکه آیا درگاه در لیست WooCommerce وجود دارد
            if (isset($wc_gateways[$gateway_id])) {
                $gateway_instance = $wc_gateways[$gateway_id];
            }
        }
        
        // اگر در لیست WooCommerce نیست، سعی کن از طریق کلاس instantiate کنی
        if (!$gateway_instance && class_exists($class_name_to_check)) {
            try {
                $gateway_instance = new $class_name_to_check();
                
                // اطمینان از لود شدن تنظیمات درگاه
                if (method_exists($gateway_instance, 'init_settings')) {
                    $gateway_instance->init_settings();
                }
            } catch (Exception $e) {
                // اگر ساخت نمونه با خطا مواجه شد، صرفاً نادیده بگیر
                error_log('IranDargah: Failed to instantiate gateway ' . $gateway_id . ': ' . $e->getMessage());
                $gateway_instance = null;
            }
        }
        
        // اگر درگاه instance موفق بود، آن را اضافه کن
        if ($gateway_instance) {
            // اطمینان از فعال بودن درگاه
            if (method_exists($gateway_instance, 'update_option')) {
                $gateway_instance->update_option('enabled', 'yes');
            }
            if (property_exists($gateway_instance, 'enabled')) {
                $gateway_instance->enabled = 'yes';
            }
            
            // اطمینان از اینکه درگاه available است
            if (method_exists($gateway_instance, 'is_available')) {
                // اگر متد is_available false برمی‌گرداند، آن را override کن
                // اما اول سعی کن تنظیمات را درست کنی
                $option_name = "woocommerce_{$gateway_id}_settings";
                $gateway_settings = get_option($option_name, []);
                $gateway_settings['enabled'] = 'yes';
                update_option($option_name, $gateway_settings);
            }
            
            // اضافه کردن به لیست available
            // استفاده از ID درگاه به عنوان کلید
            $gateway_key = method_exists($gateway_instance, 'get_id') ? $gateway_instance->get_id() : $gateway_id;
            $available[$gateway_key] = $gateway_instance;
        }
    }

    // درگاه‌هایی که در لیست ایران‌درگاه نیستند را دست نخورده نگه دار (مثل درگاه‌های دیگر یا روش‌های پرداخت سفارشی)

    return $available;
}, 20);

/**
 * Override متد is_available() برای درگاه‌های ایران‌درگاه
 * اگر درگاه در داشبورد فعال باشد، همیشه true برگرداند (حتی اگر تنظیمات ناقص باشد)
 */
add_filter('woocommerce_gateway_is_available', function($is_available, $gateway) {
    // فقط در فرانت‌اند (نه در admin)
    if (is_admin()) {
        return $is_available;
    }
    
    // اگر درگاه از قبل available است، نیازی به تغییر نیست
    if ($is_available) {
        return $is_available;
    }
    
    $gateway_ids = irandargah_get_gateway_ids();
    
    // دریافت ID درگاه
    $gateway_id = '';
    if (is_object($gateway)) {
        if (method_exists($gateway, 'get_id')) {
            $gateway_id = $gateway->get_id();
        } elseif (isset($gateway->id)) {
            $gateway_id = $gateway->id;
        }
    }
    
    // بررسی اینکه آیا این درگاه در لیست ایران‌درگاه است
    if (empty($gateway_id) || !in_array($gateway_id, $gateway_ids, true)) {
        return $is_available;
    }
    
    // بررسی وضعیت فعال بودن درگاه در داشبورد ایران‌درگاه
    $statuses = get_option('irandargah_gateway_statuses', []);
    $is_enabled_in_dashboard = isset($statuses[$gateway_id]) && (int)$statuses[$gateway_id] === 1;
    
    // اگر درگاه در داشبورد فعال است، آن را available کن
    if ($is_enabled_in_dashboard) {
        return true;
    }
    
    return $is_available;
}, 10, 2);
